/************************************************************************ 
  Copyright 2011-2012, IceJS Team. All rights reserved.
  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:
  
      * Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.
      * Redistributions in binary form must reproduce the above
        copyright notice, this list of conditions and the following
        disclaimer in the documentation and/or other materials provided
        with the distribution.
      * Neither the name of Update Freezer Team nor the names of its
        contributors may be used to endorse or promote products derived
        from this software without specific prior written permission.
  
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***********************************************************************/

/**
 * @fileOverview
 * This is a wrapper for Shell functions.
 *
 * Attribute like @[description] used here to generate help.chm file.
 */

if (typeof(Util) == "undefined")
	Util = {};

/**
 * Object that represent Shell functions.
 * @class Util.Shell
 */
Util.Shell = 
{
	/**
	 * Get environment variable.
	 * @param name Variable name.
	 * @returns Variable value or false.
	 */
	GetEnvironmentVariable: function(/**String*/ name) /**String*/
	{
		return _GetEnvironmentVariable(name);
	},
	
	/**
	 * Set environment variable value.
	 * @param name Variable name.
	 * @param value Variable value.
	 * @param target
	 * 0 - set it to Process
	 * 1 - set it to Machine
	 * 2 - set it to User
	 * @returns Boolean. True if success.
	 */
	SetEnvironmentVariable: function(/**String*/ name, /**String*/ value, /**Number*/ target) /**Boolean*/
	{
		return _SetEnvironmentVariable(name, value, target);
	},

	/**
	 * Expand environment string.
	 * @param str String with environment strings abbreviations.
	 * @returns expanded string
	 */
	ExpandEnvironmentString: function(/**String*/ str) /**String*/
	{
		return _ExpandEnvironmentString(str);
	},
	
	/**
	 * Is Windows 64 bit.
	 * @returns Boolean. True if 64
	 */
	IsWin64: function() /**Boolean*/
	{
		return _IsWin64();
	},

	/**
	 * Get OS version
	 * @returns Object with fields majorVersion, minorVersion, buildNumber, servicePackMajor, servicePackMinor
	 */
	GetOSVersion: function() /**Object*/
	{
		/*
		Windows 1.0 	1.04
		Windows 2.0 	2.11
		Windows 3.0 	3
		Windows NT 3.1 	3.10.528
		Windows for Workgroups 3.11 	3.11
		Windows NT Workstation 3.5 	3.5.807
		Windows NT Workstation 3.51 	3.51.1057
		Windows 95 	4.0.950
		Windows NT Workstation 4.0 	4.0.1381
		Windows 98 	4.1.1998
		Windows 98 Second Edition 	4.1.2222
		Windows Me 	4.90.3000
		Windows 2000 Professional 	5.0.2195
		Windows XP 	5.1.2600
		Windows Vista 	6.0.6000
		Windows 7 	6.1.7600
		*/
		return _GetOSVersion();
	},

	/**
	 * Is Windows version is equal or later than Vista.
	 * @returns Boolean. True if later
	 */
	IsVistaOrLater: function() /**Boolean*/
	{
		var versionInfo = _GetOSVersion();
		return versionInfo.majorVersion > 5;
	},
	
	/**
	 * Get Internet Explorer major version.
	 * @returns Number.
	 */
	GetIEMajorVersion: function() /**Number*/
	{
		var value = Util.Registry.GetKeyValueString(Util.Registry.HKLM, "SOFTWARE\\Microsoft\\Internet Explorer", "Version");
		if (value !== false)
		{
			var version = value.split('.')[0];
			return parseInt(version);
		}
		return 7;
	},
	
	/**
	 * Get Application data folder path.
	 * @returns path to folder.
	 */
	GetLocalAppDataFolder: function() /**String*/
	{
		var userprofile = _GetEnvironmentVariable("USERPROFILE");
		if (this.IsVistaOrLater())
		{
			// VISTA, WIN7
			return userprofile + "\\AppData\\Local";
		}
		else
		{
			// XP	
			return userprofile + "\\Local Settings\\Application Data";	
		}
	},
	
	/**
	 * Get Desktop folder path.
	 * @returns path to folder.
	 */
	GetDesktopFolder: function() /**String*/
	{
		return _ExpandEnvironmentString(Util.Registry.GetKeyValueString(Util.Registry.HKCU, "Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\User Shell Folders", "Desktop"));
	},

	/**
	 * Get Program data folder path.
	 * @returns path to folder.
	 */
	GetProgramDataFolder: function() /**String*/
	{
		return Util.Shell.GetEnvironmentVariable("ALLUSERSPROFILE");
	},
	
	/**
	 * Run executable and wait for result
	 * @param cmdLine Command line.
	 * @param elevatedPrivileges Elevate privileges or not.
	 * @returns process exit code
	 */
	Run: function(/**String*/ cmdLine, /**Boolean*/ elevatedPrivileges) /**Number*/
	{
		return _RunProcess(cmdLine, elevatedPrivileges||false);
	},
	
	/**
	 * Create new process
	 * @param cmdLine Command line.
	 * @param elevatedPrivileges Elevate privileges or not.
	 * @returns PID of newly created process. 0 if error
	 */
	CreateProcess: function(/**String*/ cmdLine, /**Boolean*/ elevatedPrivileges) /**Number*/
	{
		return _CreateProcess(cmdLine, elevatedPrivileges||false);
	},
	
	/**
	 * Execute new process
	 * @param file Path to file.
	 * @param parameters
	 * @returns PID of newly created process. 0 if error
	 */
	Execute: function(/**String*/ file, /**String*/ parameters) /**Number*/
	{
		parameters = parameters||"";
		return _ShellExecute(file, parameters);
	},

	/**
	 * Is dll loadable.
	 * @param dllName Path to dll.
	 * @returns Boolean. True if success
	 */
	IsDllLoadable: function(/**String*/ dllName) /**Boolean*/
	{
		return _CheckDllLoadable(dllName);
	},

	/**
	 * Sleep
	 * @param millis Time to sleep in milliseconds.
	 * @returns Boolean. True if success
	 */
	Sleep: function(/**Number*/ millis) /**Boolean*/
	{
		return _Sleep(millis);
	}
}