/************************************************************************ 
  Copyright 2011-2012, IceJS Team. All rights reserved.
  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:
  
      * Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.
      * Redistributions in binary form must reproduce the above
        copyright notice, this list of conditions and the following
        disclaimer in the documentation and/or other materials provided
        with the distribution.
      * Neither the name of Update Freezer Team nor the names of its
        contributors may be used to endorse or promote products derived
        from this software without specific prior written permission.
  
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***********************************************************************/
 
/**
* @fileOverview
* This is a wrapper for registry functions.
*
* Attribute like @[description] used here to generate help.chm file.
*/

if (typeof(Util) == "undefined")
	Util = {};

/**
 * Object that contain functions to work with registry.
 * @class Util.Registry
 */
Util.Registry = 
{
	/**
	 * Section HKEY_CLASSES_ROOT
	 * @type String
	 * @field
	 */
	HKCR: "HKEY_CLASSES_ROOT",
	/**
	 * Section HKEY_CURRENT_USER
	 * @type String
	 * @field
	 */
	HKCU: "HKEY_CURRENT_USER",
	/**
	 * Section HKEY_LOCAL_MACHINE
	 * @type String
	 * @field
	 */
	HKLM: "HKEY_LOCAL_MACHINE",
	/**
	 * Section HKEY_USERS
	 * @type String
	 * @field
	 */
	HKU: "HKEY_USERS",
	/**
	 * Section HKEY_PERFORMANCE_DATA
	 * @type String
	 * @field
	 */
	HKPD: "HKEY_PERFORMANCE_DATA",
	/**
	 * Section HKEY_CURRENT_CONFIG
	 * @type String
	 * @field
	 */
	HKCC: "HKEY_CURRENT_CONFIG",

	/**
	 * Get registry key value as string.
	 * @param section Registry section to use
	 * @param keyPath Path to key.
	 * @param keyName Key name.
	 * @returns Value as String
	 */
	GetKeyValueString: function(/**String*/ section, /**String*/ keyPath, /**String*/ keyName) /**String*/ 
	{
		return _QueryRegistryKeyValueString(section, keyPath, keyName);
	},

	/**
	 * Get registry key value as number.
	 * @param section Registry section to use
	 * @param keyPath Path to key. 
	 * @param keyName Key name.
	 * @returns Value as Number
	 */
	GetKeyValueDWORD: function(/**String*/ section, /**String*/ keyPath, /**String*/ keyName) /**Number*/
	{
		return _QueryRegistryKeyValueDWORD(section, keyPath, keyName);
	},

	/**
	 * Set number value to registry key.
	 * @param section Registry section to use
	 * @param path Path to key.
	 * @param valName Key name.
	 * @param valData Value to set.
	 * @returns Boolean True if success.
	 */
	SetKeyValueDWORD: function(/**String*/ section, /**String*/ path, /**String*/ valName, /**Number*/ valData) /**Boolean*/
	{
		return _SetRegistryKeyValueDWORD(section, path, valName, valData);
	},

	/**
	 * Set string value to registry key. If key does not exist, will create it.
	 * @param section Registry section to use
	 * @param path Path to key.
	 * @param valName Key name.
	 * @param valData Value to set.
	 * @returns Boolean True if success.
	 */
	SetKeyValueString: function(/**String*/ section, /**String*/ path, /**String*/ valName, /**String*/ valData) /**Boolean*/
	{
		return _SetRegistryKeyValueString(section, path, valName, valData);
	},

	/**
	 * Delete registry key value.
	 * @param section Registry section to use
	 * @param path Path to registry key.
	 * @param valName Registry key value name.
	 * @returns Boolean True if success.
	 */
	DeleteKeyValue: function(/**String*/ section, /**String*/ path, /**String*/ valName) /**Boolean*/
	{
		return _DeleteRegistryValue(section, path, valName);
	},
	
	/**
	 * Delete registry key.
	 * @param section Registry section to use
	 * @param path Path to registry key.
	 * @returns Boolean True if success.
	 */
	DeleteKey: function(/**String*/ section, /**String*/ path) /**Boolean*/
	{
		return _DeleteRegistryKey(section, path);
	},

	/**
	 * Is key value exists.
	 * @param section Registry section to use
	 * @param path Path to registry key.
	 * @param valName Key value name.
	 * @returns Boolean True if exists.
	 */
	IsKeyValueExists: function(/**String*/ section, /**String*/ path, /**String*/ valName) /**Boolean*/
	{
		return _IsRegistryKeyValueExists(section, path, valName);
	},

	/**
	 * Is key exists.
	 * @param section Registry section to use
	 * @param path Path to registry key.
	 * @returns Boolean True if exists.
	 */
	IsKeyExists: function(/**String*/ section, /**String*/ path) /**Boolean*/
	{
		return _IsRegistryKeyExists(section, path);
	},
	
	/**
	 * Ensure key exists. Creates all necessary parents in registry path.
	 * @param section Registry section to use
	 * @param path Path to registry key.
	 * @returns Boolean True if exists.
	 */
	EnsureKeyExists: function(/**String*/ section, /**String*/ path) /**Boolean*/
	{
		if(!_IsRegistryKeyExists(section, path))
		{
			if(!_CreateRegistryKey(section, path))
				return false;
		}
		return true;
	},

	/**
	 * Enum children key names.
	 * @param section Registry section to use
	 * @param path Path to registry key.
	 * @returns Array of string.
	 */
	EnumChildrenKeyNames: function(/**String*/ section, /**String*/ path) /**Array*/
	{
		return _RegEnumChildrenKeyNames(section, path);
	},

	/**
	 * Enum children key value names.
	 * @param section Registry section to use
	 * @param path Path to registry key.
	 * @returns Array of string.
	 */
	EnumChildrenValues: function(/**String*/ section, /**String*/ path) /**Array*/
	{
		return _RegEnumChildrenValues(section, path);
	}
}