/************************************************************************ 
  Copyright 2011-2012, IceJS Team. All rights reserved.
  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:
  
      * Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.
      * Redistributions in binary form must reproduce the above
        copyright notice, this list of conditions and the following
        disclaimer in the documentation and/or other materials provided
        with the distribution.
      * Neither the name of Update Freezer Team nor the names of its
        contributors may be used to endorse or promote products derived
        from this software without specific prior written permission.
  
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***********************************************************************/

/**
 * @fileOverview
 * This is a wrapper for LightUI. Contain UI functions.
 *
 * Attribute like @[description] used here to generate help.chm file.
 */

if (typeof(Util) == "undefined")
	Util = {};

/**
 * Object that represent LightUI functions.
 * @class Util.LightUI
 */
Util.LightUI = 
{
	/**
	 * Eval string.
	 * @param str String to eval
	 */
	Eval: function(/**String*/ str)
	{
		_LightUIEval(str);
	},

	/**
	 * Show alert message.
	 * @param str String to show.
	 */
	Alert: function(/**String*/ str)
	{
		_LightUIEval("alert(" + JSON.stringify(str) + ")");
	},

	/**
	 * Show confirm dialog
	 * @param str String to show
	 * @returns String
	 */
	Confirm: function(str) /**String*/
	{
		var str = "var res = confirm(" + JSON.stringify(str) + "); document.getElementById('#tmplogbuff').innerHTML = res ? 'yes' : 'no';";
		Util.LightUI.Eval(str);
		var res = _LightUIGetElementInnerText('#tmplogbuff');
		return res;
	},

	/**
	 * Show dialog
	 * @param title
	 * @param entryPoint Path to html document.
	 */
	ShowDialog: function (/**String*/ title, /**String*/ entryPoint)
	{
		_LightUIShowDialog(title, entryPoint);
	},

	/**
	 * Close dialog
	 */
	Close: function()
	{
		_LightUIClose();
	},
	
	/**
	 * Set inner text in html document element.
	 * @param id Id of html element.
	 * @param text Text to set.
	 */
	SetElementInnerText: function(/**String*/ id, /**String*/ text)
	{
		_LightUISetElementInnerText(id, text);
	},

	/**
	 * Get inner text of html document element.
	 * @param id Id of html element.
	 * @returns text.
	 */
	GetElementInnerText: function(/**String*/ id) /**String*/
	{
		return _LightUIGetElementInnerText(id);
	},

	/**
	 * Set main dialog size.
	 * @param width to set.
	 * @param height to set.
	 * @returns True if success
	 */
	SetDialogSize: function(/**Number*/ width, /**Number*/ height) /**Boolean*/
	{
		return _LightUISetDialogSize(width, height);
	},
	
	/**
	 * Set fixed main dialog size. Same as SetDialogSize + user has no possibility to change size via mouse
	 * @param width to set.
	 * @param height to set.
	 * @returns True if success
	 */
	SetFixedDialogSize: function(/**Number*/ width, /**Number*/ height) /**Boolean*/
	{
		return _LightUISetFixedDialogSize(width, height);
	},
	
	/**
	 * Show or hide Maximize Box (right top corner of window, near to button close)
	 * @param show Use 'true' to show icon.
	 * @returns True if success
	 */
	ShowMaximizeBox: function(/**Boolean*/ show) /**Boolean*/
	{
		return _LightUIShowMaximizeBox(show);
	},

	/**
	 * Show or hide Minimize Box (right top corner of window, near to button close)
	 * @param show Use 'true' to show icon.
	 * @returns True if success
	 */
	ShowMinimizeBox: function(/**Boolean*/ show) /**Boolean*/
	{
		return _LightUIShowMinimizeBox(show);
	},

	/**
	 * Set main dialog top left point position on screen.
	 * @param left to set.
	 * @param top to set.
	 * @returns Boolean. True if success
	 */
	SetDialogPosition: function(/**Number*/ left, /**Number*/ top) /**Boolean*/
	{
		return _LightUISetDialogPosition(left, top);
	},

	/**
	 * Get current dialog size.
     * @returns Object with fields like {x:0, y:0, w:0, h:0} 
     */
	GetDialogSize: function() /**Object*/
	{
		return _LightUIGetDialogSize();
	},
	
    /**
	 * Enum displays resolution and position.
     * @returns Array of objects, length depends on available monitors [{x:0, y:0, w:0, h:0}, {x:0, y:0, w:0, h:0}]
     */
	GetDisplayInfo: function() /**Array*/
	{
		return _LightUIGetDisplayInfo();
	},
	
	/**
	 * Center main dialog.
	 * @returns Boolean
	 */
	CenterDialog: function() /**Boolean*/
	{
		return _LightUICenterDialog();
	},
	
	/** ShowWindow contants
		#define SW_HIDE             0
		#define SW_SHOWNORMAL       1
		#define SW_NORMAL           1
		#define SW_SHOWMINIMIZED    2
		#define SW_SHOWMAXIMIZED    3
		#define SW_MAXIMIZE         3
		#define SW_SHOWNOACTIVATE   4
		#define SW_SHOW             5
		#define SW_MINIMIZE         6
		#define SW_SHOWMINNOACTIVE  7
		#define SW_SHOWNA           8
		#define SW_RESTORE          9
		#define SW_SHOWDEFAULT      10
		#define SW_FORCEMINIMIZE    11
		#define SW_MAX              11	
	*/
	
	/**
	 * Minimize console window.
	 * @returns Boolean. True if success
	 */
	MinimizeConsoleWindow: function() /**Boolean*/
	{
		return _LightUIShowConsoleWindow(6);
	},
	
	/**
	 * Hide console window.
	 * @returns Boolean. True if success
	 */
	HideConsoleWindow: function() /**Boolean*/
	{
		return _LightUIShowConsoleWindow(0);
	},
	
	/**
	 * Bring main dialog window to front.
	 * @returns Boolean. True if success
	 */
	BringToFront: function() /**Boolean*/
	{
		return _LightUIBringToFront();
	},
	
	/**
	 * Set wait cursor.
	 * @returns Boolean. True if success
	 */
	WaitCursor: function() /**Boolean*/
	{
		return _WaitCursor();
	},
	
	/**
	 * Set system default cursor.
	 * @returns Boolean. True if success
	 */
	DefaultCursor: function() /**Boolean*/
	{
		return _DefaultCursor();
	},
	
	/**
	 * Create shortcut on desktop.
	 * @returns Boolean. True if success
	 */
	CreateDesktopShortcut: function() /**Boolean*/
	{
		var info = Util.IO.ReadJsonFromFile("info.json");
		if (info)
		{
			var lnkPath = Util.Shell.GetDesktopFolder() + "\\" + info.shortcutText + ".lnk";
			if (!info.shortcutInstalled)
			{
				// do not bug user with this in the future
				info.shortcutInstalled  = true;
				Util.IO.WriteJsonToFile("info.json", info)
				
				var res = "yes";
				var shortcutExists = Util.IO.CheckFileExists(lnkPath);
				if (!shortcutExists)
				{
					res = Util.LightUI.Confirm(SF("UI.InstallShortcutConfirm"));
				}
				if (res == "yes")
				{
				 	Util.IO.DeleteFile(lnkPath);
					var workingDirectory = Util.IO.GetCurrentDirectory();//Util.LightUI.GetInstallationPath(info);
					var icejsPath = Util.IO.CombinePath(workingDirectory, "Bin\\icejs.exe");
					var cmdLineParams = info.entryPoint;
					var iconPath =  Util.IO.CombinePath(workingDirectory, "Resources\\" + info.scenarioName + ".ico");
					Util.IO.CreateShortcut(icejsPath, cmdLineParams, workingDirectory, lnkPath, info.scenarioName, iconPath, 0);
					
					if (shortcutExists)
					{
						// the shortcut was updated
						return false;
					}
					
					return true;
				}
			}
		}	
		return false;
	},
	
	/**
	 * Get installation path.
	 * @param info Object with information, see info.json file.
	 * @returns String.
	 */
	GetInstallationPath: function(/**Object*/ info) /**String*/
	{
		return Util.IO.CombinePath(Util.Shell.GetProgramDataFolder(), "Application Data\\IceJS\\" + info.scenarioName + "\\" + info.version);
	},
	
	/**
	 * Set dialog icon from filename and index.
	 * @param fileName Path to file with icon.
	 * @param [index] Icon index, 0-based.
	 * @returns Boolean. True if success
	 */
	SetIcon: function(/**String*/ fileName, /**Number*/ index) /**Boolean*/
	{
		if(typeof(index) == 'undefined' || index < 0)
		{
			index = 0;
		}
		return _LightUISetIcon(fileName, index);
	},

	/**
	 * Set dialog title.
	 * @param t Title to set.
	 * @returns Boolean. True if success
	 */
	SetTitle: function(/**String*/ t) /**Boolean*/
	{
		return _LightUISetTitle(t);
	},
	
	/**
	 * Set focus to dialog.
	 */
	SetFocus: function()
	{
		_LightUIEval("document.body.focus();");
	},
	
	/**
	 * Do events. Use it in time-consuming operations to avoid window screen Not responding
	 * @param ms Total time to wait.
	 * @param diff Maximum interval to sleep.
	 */
	DoEvents: function(/**Number*/ ms, /**Number*/ diff)
	{
		ms = ms||50;
		diff = diff||50;
		_DoEvents(ms, diff);
	}	
}