/************************************************************************ 
  Copyright 2011-2012, IceJS Team. All rights reserved.
  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:
  
      * Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.
      * Redistributions in binary form must reproduce the above
        copyright notice, this list of conditions and the following
        disclaimer in the documentation and/or other materials provided
        with the distribution.
      * Neither the name of Update Freezer Team nor the names of its
        contributors may be used to endorse or promote products derived
        from this software without specific prior written permission.
  
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***********************************************************************/
 
if (typeof(g_executeInBatch) == "undefined")
{
	Include("Wrappers/Common.js")
	Include("Controllers/Controller.js");
}

if (typeof(Scenario) == "undefined")
	Scenario = {};


Scenario.AdobeUpdater = 
{
	name: 			"Adobe.Updater",
	displayName:    "Adobe Acrobat & Reader",
	version: 		"1.3",
	description: 	"Checks if Adobe update is enabled and allows to disable and enable update facilities",
	link:			"http://www.updatefreezer.org/index.php?id=2",
	icon:			"Resources/Adobe.Updater.png",
	actions:		["query", "backup", "restore", "enable", "disable"],

	programs:		[{path: "%PROGRAMFILES%\\Adobe"}],
	inherits: 		[ScenarioController],
	
//#region Implementation
	adobeProductsDataFileName: "Adobe.Products.json",
	regRoot: "SOFTWARE\\Policies\\Adobe",
	lockKeyName: "FeatureLockDown",
	rk: "bUpdater",
	supportedProducts: ["Acrobat", "Acrobat Reader"],
	
	_isProductSupported: function(productName)
	{
		for(var i = 0; i < this.supportedProducts.length; i++)
		{
			if (this.supportedProducts[i] == productName)
			{
				return true;
			}
		}
		return false;
	},
	
	IsProgramInstalledExt: function()
	{
		// check that required registry keys exist
		var someProductsExist = false;
		for(var i = 0; i < this.supportedProducts.length; i++)
		{
			if (Util.Registry.IsKeyExists(Util.Registry.HKLM, "SOFTWARE\\Policies\\Adobe\\" + this.supportedProducts[i]))
			{
				someProductsExist = true;
			}
		}
		
		return someProductsExist;
	},

	_EnumerateProducts: function()
	{
		var p = [];
		//To disable the Updater and remove the Updater configuration UI in the Preferences panel:
		//HKEY_LOCAL_MACHINE\SOFTWARE\Policies\Adobe\{product}\{version}\FeatureLockdown
		//Create the new DWord bUpdater.
		//Set the value to 0.
		var products = Util.Registry.EnumChildrenKeyNames(Util.Registry.HKLM, this.regRoot);
		for(var i = 0; i < products.length; i++)
		{
			var productName = products[i].name;
			
			if (!this._isProductSupported(productName))
			{
				continue;
			}
			
			var versions = Util.Registry.EnumChildrenKeyNames(Util.Registry.HKLM, this.regRoot + "\\" + productName);
			for(var v = 0; v < versions.length; v++)
			{
				var version = versions[v].name;
				if(Util.Registry.IsKeyExists(Util.Registry.HKLM, this.regRoot + "\\" + productName + "\\" + version + "\\" + this.lockKeyName))
				{
					var product = {"name": productName, "version": version, "enabled": true};
					// check enable
					var q = Util.Registry.GetKeyValueDWORD(Util.Registry.HKLM, 
														   this.regRoot + "\\" + productName + "\\" + version + "\\" + this.lockKeyName,
														   this.rk);
					if(q === 0)
					{
						product.enabled = false;
					}
					p.push(product);
				}
			}
		}
		return p;
	},

	QueryAdobeProducts: function()
	{
		var result = {
			products:[],
			hasProducts: function()
			{
				return this.products.length > 0;
			},
			hasEnabled: function()
			{
				for(var i = 0; i < this.products.length; i++)
				{
					if (this.products[i].enabled)
						return true;
				}
				return false;
			}
		};
		
		result.products = this._EnumerateProducts();
		var dataFileName =  Util.IO.CombinePath(this.dataFolderPath, this.adobeProductsDataFileName);
		Util.IO.WriteJsonToFile(dataFileName, result);
		
		if (result.hasProducts())
		{
			if (result.hasEnabled())
			{
				this.updaterEnabled = true;
				queryResult.push({msg: S("Adobe.Updater.ProductsEnabled")});
			}
			else
				queryResult.push({msg: S("Adobe.Updater.ProductsDisabled")});
		}
		else
		{
			queryResult.push({msg: S("Adobe.Updater.ProductsNotFound")});
		}
		return result;
	},
	
	ChangeAdobeProducts: function(enable)
	{
		var dataFileName =  Util.IO.CombinePath(this.dataFolderPath, this.adobeProductsDataFileName);
		var p = Util.IO.ReadJsonFromFile(dataFileName);
		if (p && p.products)
		{
			for(var i = 0; i < p.products.length; i++)
			{
				var product = p.products[i];
				var et = enable;
				if (enable === undefined)
					et = product.enabled;
					
				Util.Registry.SetKeyValueDWORD(Util.Registry.HKLM, this.regRoot + "\\" + product.name + "\\" + product.version + "\\" + this.lockKeyName, this.rk, et ? 1 : 0);
			}
		}
	},
//#endregion	

	
//#region Actions
	Query: function()
	{
		if (!this.CheckProgramInstalled())
		{
			return false;
		}	
		
		this.QueryAdobeProducts();
		this.LogUpdaterStatus();
	},
	
	Disable: function()
	{
		this.ChangeAdobeProducts(false);
	},
	
	Enable: function()
	{
		this.ChangeAdobeProducts(true);
	},
	
	Backup: function()
	{
		this.QueryAdobeProducts();
	},
	
	Restore: function()
	{
		this.ChangeAdobeProducts();
	}
//#endregion	
	

};


Util.Scenario.InitializeScenario(Scenario.AdobeUpdater);

if (typeof(g_executeInBatch) == "undefined")
{
	Util.Locale.Load();
	Execute(Scenario.AdobeUpdater);
}

Scenario.AdobeUpdater.OutputDependencies();
